"use client";

import { motion, AnimatePresence } from "framer-motion";
import { usePathname } from "next/navigation";

export default function PageTransition({ children }: { children: React.ReactNode }) {
    const pathname = usePathname();

    return (
        <>
            {/* Page content with smooth fade */}
            <motion.div
                key={pathname}
                initial={{ opacity: 0, y: 30 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{
                    duration: 0.4,
                    ease: [0.25, 0.46, 0.45, 0.94]
                }}
            >
                {children}
            </motion.div>

            {/* Enhanced center transition indicator */}
            <AnimatePresence>
                <motion.div
                    key={`overlay-${pathname}`}
                    initial={{
                        scale: 0,
                        opacity: 0,
                        rotate: 0
                    }}
                    animate={{
                        scale: 0,
                        opacity: 0,
                        rotate: 0
                    }}
                    exit={{
                        scale: [0, 1, 1.2, 0],
                        opacity: [0, 0.8, 0.6, 0],
                        rotate: [0, 180, 360, 360]
                    }}
                    transition={{
                        duration: 0.8,
                        ease: [0.34, 1.56, 0.64, 1],
                        times: [0, 0.3, 0.7, 1]
                    }}
                    className="fixed top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 z-50 pointer-events-none"
                >
                    {/* Outer ring */}
                    <div className="relative">
                        <div className="w-12 h-12 border-2 border-primary-500/30 dark:border-gray-600 rounded-full animate-pulse" />

                        {/* Inner dot */}
                        <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-3 h-3 bg-gradient-to-r from-primary-500 to-secondary-500 dark:from-gray-600 dark:to-gray-400 rounded-full shadow-lg" />

                        {/* Ripple effect */}
                        <motion.div
                            initial={{ scale: 0, opacity: 0.8 }}
                            animate={{ scale: 2, opacity: 0 }}
                            transition={{ duration: 0.6, repeat: Infinity }}
                            className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-12 h-12 border border-primary-400/40 dark:border-primary-300/40 rounded-full"
                        />
                    </div>
                </motion.div>
            </AnimatePresence>

            {/* Subtle backdrop blur during transition */}
            <AnimatePresence>
                <motion.div
                    key={`backdrop-${pathname}`}
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 0 }}
                    exit={{ opacity: [0, 0.3, 0] }}
                    transition={{
                        duration: 0.8,
                        times: [0, 0.5, 1]
                    }}
                    className="fixed inset-0 bg-white/5 dark:bg-black/10 backdrop-blur-[1px] z-40 pointer-events-none"
                />
            </AnimatePresence>
        </>
    );
}
